///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _error_h
#define _error_h

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <sstream>
#include <string>
#include <vector>
#include <exception>


const int MAX_ERROR_MSG = 1024;


// default static description of error categories
struct NullCat
{
  static char const *category(int) {return "[unspecified error type]";}
};


// generic exception class
template<typename T = NullCat>
class Error: public std::exception
{
  int type_;
  std::string info_;
  std::string message_;

protected:
  void make_error_msg(int t, char const *loc = 0, char const *info = 0)
  {
    std::ostringstream s;
    info_ = (info && info[0])? info: "unspecified failure";
    s << T::info(t);
    s << " in \"" << (loc && loc[0]? loc: "unknown location");
    s << "\"\n>>>> " << info_;
    message_ = s.str();
  }

public:
  Error(int t = -1): type_(t) {make_error_msg(t);}
  Error(int t, char const* loc, char const *fmt, ...): type_(t)
  {
    va_list ap;
    va_start(ap, fmt);

    if (fmt)
    { // kludge to process formatted printout of unknown size
      std::vector<char> tmp;
      size_t sz;
      for (sz = strlen(fmt)*2; sz < MAX_ERROR_MSG; sz *= 2)
      {
	tmp.resize(sz, 0);
	vsnprintf(&tmp[0], tmp.size(), fmt, ap);
	if (*(tmp.end()-1) == 0)
	  break; // printout fits in buffer (including terminating '\0')
      }
      make_error_msg(t, loc, &tmp[0]);
    }
    else make_error_msg(t, loc);
  }
  virtual ~Error() throw() { }

  // error category (number >= 0)
  virtual int errnum() const {return type_;}
  // error category description
  virtual char const *category() const {return T::info(type_);}
  // brief error description
  virtual char const *info() {return info_.c_str();}
  // full description as customary for std::exception
  virtual char const *what() const throw() {return message_.c_str();}
};

#endif // _error_h
