///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _modbus_request_h
#define _modbus_request_h

#include "copy.hh"


namespace Comm
{
  namespace Modbus
  {
    // classes
    template<Protocol P = RTU> class Request
    {
    public:
      inline int station() {return buffer[0];}
      inline Function function() {return Function(buffer[1]);}

    protected:
      std::vector<uint8_t> buffer; // packet data buffer
      const PktStruct *ps;         // request structure descriptor
      int seq;                     // item sequence number (header and payload counted separately)
      bool ready, inhdr;           // packet assembled flag, in-header assembly flag
      // check whether function # is valid
      Function checkfn(Function fn)
      {
	if (checkFn(fn)) return fn;
	throw Error(PROTOCOL, "Modbus::Request::checkfn", "invalid function number");
      }
      // figure out the next item size
      int itemsize();
      // append item to the end of message
      template<typename T> void append(T item);
      // callable wrapper for byte copying
      //struct ByteCopy
      //{
      //  Request *parent;
      //  ByteCopy(Request *p): parent(p) { }
      //  void operator()(uint8_t item) {parent->append(item);}
      //} copyf;

    public:
      Request(int st, Function fn) : ps(&(PktParams[checkfn(fn)].req))//, copyf(this)
      {
	if ((fn > LASTFN) || !PktParams[fn].req.hdr.len)
	  throw Error(PROTOCOL, "Modbus::Request::checkfn", "invalid function");
	buffer.reserve(P == RTU? RTU_INIT_BUF: ASCII_INIT_BUF);
	append<uint8_t>(st & 0xff);
	append<uint8_t>(fn);
	ready = false; inhdr = true; seq = 1;
      }

      // clear request data (sans station, function and their dependent parameters)
      void reset()
      {
	buffer.resize(2); // station and function are fixed
	ready = false; inhdr = true; seq = 1;
      }
      // insert datum spanning potentially multiple slots in the packet
      template<class T> void put(T item);
      // insert datum into exactly single slot of the target packet
      template<class T> void put1(T item)
      {
	switch (itemsize())
	{
	case 1:
	  append(static_cast<uint8_t>(item));
	  break;
	case 2:
	  append(static_cast<uint16_t>(item));
	  break;
	default:
	  throw Error(PROTOCOL, "Modbus::Request::put1", "unsupported datum size");
	}
      }
      uint8_t& operator[](int i)
      {
	if (ready) return buffer[i];
	throw Error(PROTOCOL, "Modbus::Request::[]", "attempted access to non-finalized buffer");
      }
      void finalize();

      inline int bytes() {return buffer.size();}
    };
  

    // specializations
    template<> template<> inline void Request<RTU>::append(uint8_t item) {buffer.push_back(item);}
    template<> template<> inline void Request<RTU>::append(uint16_t item)
    {
      append<uint8_t>(item >> 8);
      append<uint8_t>(item & 0xff);
    }

    template<> int Request<RTU>::itemsize();
    template<> void Request<RTU>::finalize();

    template<> template<class T> void Request<RTU>::put(T item)
    {
      if (inhdr)
      { // header fields may not be aggregated, hence mapping to put1()
	put1(item);
	return;
      }
      int isz = itemsize();
      Copier<Request> cf(this, &Request::append<uint8_t>);
      switch (isz)
      {
      case 1:
	{
	  Append<T,1,Copier<Request> >(cf).copy(item);
	}
	break;
      case 2:
	{
	  Append<T,2,Copier<Request> >(cf).copy(item);
	}
	break;
      default:
	throw Error(PROTOCOL, "Modbus::Request::put", "unsupported word size");
      }
    }
  }
}

#endif // _modbus_request_h
