///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _modbus_reply_h
#define _modbus_reply_h

namespace Comm
{
  namespace Modbus
  {
    // classes
    //// reply packet header (function dependent, but w/o function storage)
    struct ReplyHeader
    {
      union {uint8_t b1; uint16_t w1; uint8_t cnt; uint8_t data; uint16_t addr; uint16_t stat; uint16_t cnt2; uint16_t subfn;};
      union {uint8_t b2; uint16_t w2; uint16_t quant; uint16_t value; uint16_t andmsk; uint16_t stat2;};
      union {uint8_t b3; uint16_t w3; uint16_t ormsk; uint16_t evcnt;};
      union {uint8_t b4; uint16_t w4; uint16_t msgcnt;};
    };

    //// reply
    template<Protocol P = RTU> class Reply
    {
    protected:
      uint16_t crc;
      uint8_t payload[P == RTU? RTU_MAX_PAYLOAD: ASCII_MAX_PAYLOAD];
      short at; // index into payload (for data readout)

    public:
      const uint8_t station;
      const Function function;
      union
      {
	ReplyHeader header;
	Exception ecode;
      };
      short psize; // payload size (bytes)

      int advance(int sz)
      {
	if (sz+at > psize)
	  throw Error(PROTOCOL, "Modbus::Reply::advance", "insufficient data in receive buffer");
	int old = at;
	at += sz;
	return old;
      }

      Reply(int st = -1, Function fn = NONE) : station(st), function(fn) {at = 0;}

      template<class T> void get(T&);
      inline bool failed() {return (function & ERROR_FLAG) != 0;}
      inline uint8_t *buffer() {return payload;}
    };


    // specializations
    template<> template<> inline void Reply<RTU>::get(uint8_t& t) {t = payload[advance(1)];}
    template<> template<> inline void Reply<RTU>::get(uint16_t& t)
    {
      int i = advance(2);
      t = (static_cast<uint16_t>(payload[i])<<8) | payload[i+1];
    }
  }
}

#endif // _modbus_reply_h
