///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#include "modbus.hh"


namespace Comm
{
  namespace Modbus
  {
    // specializations
    template<> bool Client<RTU>::validate()
    {
      // don't automatically compute CRC for CRC-containing data...
      uint16_t crclo = TCPClient::get();
      uint16_t crchi = TCPClient::get();
      return (checksum == ((crchi << 8) | crclo));
    }

    template<> Reply<RTU> *Client<RTU>::recv()
    {
      checksum = RTU_INIT_CRC;
      // minimal reply: station(1)+function(1)+error_code(1)+crc(2)
      TCPClient::recv(5);
      // function has to be fetched w/o CRC computation due to issues
      // with some devices (see below)
      uint8_t stn = get(), fn = TCPClient::get();
      // fix occasional problem in Yokogawa PR300
      if ((!fn) && (flags & PR300_NULL_FN)) fn = get();
      else checksum = crc16(&fn, 1, checksum);
      // at this point, the fn value was included in CRC computation

      // mangled messages may carry invalid function value
      if (!checkFn(Function(fn & 0x7f)))
      {
	TCPClient::resync();
	throw Error(CORRUPT, "Modbus::Client::recv");
      }

      Reply<RTU> *rep = new Reply<>(stn, Function(fn));
      if (rep->failed())
      { // received error reply
	rep->ecode = Exception(get());
      }
      else
      { // correct reply received

	// grab header first (don't count function field)
	const PktStruct& ps = PktParams[fn].rep;
	short pix = ps.cntidx, psz = ps.hdr.items[pix], pcnt = 0;

	// bytecount: w/o function (-1), but (potentially) with CRC (+2) -> +1
	TCPClient::recv(ps.hdr.len+1);
	int hf = ps.hdr.cnt;
	do
	{
	  if (--hf <= 0) break;
	  setEntry(1, ps, *rep);
	  if (--hf <= 0) break;
	  setEntry(2, ps, *rep);
	  if (--hf <= 0) break;
	  setEntry(3, ps, *rep);
	  if (--hf <= 0) break;
	  setEntry(4, ps, *rep);
	} while (0);

	// copy the payload
	TCPClient::recv(rep->psize+2); // also receive CRC
	get(rep->buffer(), rep->psize); // psize == 0 is ok here
      }
      if (!validate())
      {
	delete rep;
	rep = 0;
	TCPClient::resync();
	throw Error(CHECKSUM, "Modbus::Client::recv");
      }
      return rep;
    }

    template<> uint8_t Client<RTU>::get()
    {
      uint8_t byte = TCPClient::get();
      checksum = crc16(&byte, 1, checksum);
      return byte;
    }

    template<> void Client<RTU>::get(void *buf, int cnt)
    {
      TCPClient::get(buf, cnt);
      checksum = crc16(reinterpret_cast<uint8_t *>(buf), cnt, checksum);
    }
  }
}
