///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _modbus_h
#define _modbus_h

#include <cctype>
#include <vector>
#include "conf.hh"
#include "comm_error.hh"
#include "client.hh"

namespace Comm
{
  namespace Modbus
  {
    // Modbus functions
    enum Function
    {
      NONE = 0,

      RD_COILS = 1,
      RD_INPUTS,
      RD_HREGS,
      RD_IREGS,
      WR_COIL,
      WR_REG,
      RD_EXC_STAT,
      DIAG,

      GET_COMM_CNT = 11,
      GET_COMM_LOG,

      WR_COILS = 15,
      WR_REGS,
      DEV_ID,

      RD_FILE_REC = 20,
      WR_FILE_REC,
      WR_REG_MASK,
      RW_REGS,
      RD_FIFO,

      ENC_TRAN = 43,
      LASTFN = ENC_TRAN,

      ERROR_FLAG = 0x80
      // others ignored for now
    };

    // Modbus subfunctions
    enum SubFunction
    {
      RET_QUERY_DATA = 0,
      RET_COMM_OPT,
      RET_DIAG_REG,
      ASCII_IN_DEL,
      LISTEN_ONLY,

      CLR_CNTRS_DIAG = 10,
      RET_BUS_MSG_CNT,
      RET_BUS_COMM_ERR_CNT,
      RET_BUS_EXC_ERR_CNT,
      RET_SLV_MSG_CNT,
      RET_SLV_NORESP_CNT,
      RET_SLV_NAK_CNT,
      RET_SLV_BUSY_CNT,
      RET_BUS_CHR_OVR_CNT,

      CLR_OVR_CNTR = 20
    };

    // Modbus exceptions
    enum Exception
    {
      EXC_NONE = 0,

      EXC_ILL_FN,
      EXC_DATA_ADDR,
      EXC_DATA_VAL,
      EXC_DEV_FAIL,
      EXC_ACK,
      EXC_DEV_BUSY,
      // 7 omitted
      EXC_MEM_PARITY = 8,
      // 9 omitted
      EXC_GTWAY_PATH = 10,
      EXC_GTWAY_TGT
    };

    static const char *ExcInfo[] =
    {
      "none",
      "illegal function",
      "illegal data address",
      "illegal data value",
      "slave device failure",
      "acknowlegde",
      "slave device busy",
      "reserved (7)",
      "memory parity error",
      "reserved (9)",
      "gateway path unavailable",
      "gateway target device failed to respond"
    };

    // Modbus protocols
    enum Protocol
    {
      RTU,
      ASCII
    };

    // initial checksum values
    static const uint16_t RTU_INIT_CRC = 0xffff;
    static const uint16_t ASCII_INIT_LRC = 0;
    // max. ADU sizes
    static const int RTU_MAX_PAYLOAD = 253;
    static const int ASCII_MAX_PAYLOAD = 253;
    // guesses of average message sizes
    static const int RTU_INIT_BUF = 20;
    static const int ASCII_INIT_BUF = 32;


    // packet structure descriptions:
    //  - header variants
    struct HdrStruct {uint8_t len, cnt, items[6];};
    const HdrStruct h1 = {1, 1, {1}};
    const HdrStruct h11 = {2, 2, {1, 1}};
    const HdrStruct h12 = {3, 2, {1, 2}};
    const HdrStruct h122 = {5, 3, {1, 2, 2}};
    const HdrStruct h1221 = {6, 4, {1, 2, 2, 1}};
    const HdrStruct h1222 = {6, 4, {1, 2, 2, 2}};
    const HdrStruct h11222 = {8, 5, {1, 1, 2, 2, 2}};
    const HdrStruct h122221 = {10, 6, {1, 2, 2, 2, 2, 1}};
    //  - header
    //  - embedded count offset and adjustment
    //  - payload unit
    struct PktStruct
    {
      const HdrStruct hdr;
      short cntidx, cntadj, plunit;
    };
    const struct {PktStruct req, rep;} PktParams[] =
    {
      {{0}, {0}}, // NONE
      {{h122, 0, 0, 0}, {h11, 1, 0, 1}}, // RD_COILS
      {{h122, 0, 0, 0}, {h11, 1, 0, 1}}, // RD_INPUTS
      {{h122, 0, 0, 0}, {h11, 1, 0, 2}}, // RD_HREGS
      {{h122, 0, 0, 0}, {h11, 1, 0, 2}}, // RD_IREGS
      {{h122, 0, 0, 0}, {h122, 0, 0, 0}}, // WR_COIL
      {{h122, 0, 0, 0}, {h122, 0, 0, 0}}, // WR_REG
      {{h1, 0, 0, 0}, {h11, 0, 0, 0}},    // RD_EXC_STAT
      {{h122, 0, 0, 0}, {h122, 0, 0, 0}}, // DIAG
      {{0}, {0}}, // reserved: 9
      {{0}, {0}}, // reserved: 10
      {{h1, 0, 0, 0}, {h122, 0, 0, 0}}, // GET_COMM_CNT
      {{h1, 0, 0, 0}, {h11222, 1, -6, 1}}, // GET_COMM_LOG
      {{0}, {0}}, // reserved: 13
      {{0}, {0}}, // reserved: 14
      {{h1221, 3, 0, 1}, {h122, 0, 0, 0}}, // WR_COILS
      {{h1221, 3, 0, 2}, {h122, 0, 0, 0}}, // WR_REGS
      {{h1, 0, 0, 0}, {h11, 1, 0, 1}}, // DEV_ID
      {{0}, {0}}, // reserved: 18
      {{0}, {0}}, // reserved: 19
      {{h11, 1, 0, 1}, {h11, 1, 0, 1}}, // RD_FILE_REC
      {{h11, 1, 0, 1}, {h11, 1, 0, 1}}, // WR_FILE_REC
      {{h1222, 0, 0, 0}, {h1222, 0, 0, 0}}, // WR_REG_MASK
      {{h122221, 5, 0, 1}, {h11, 1, 0, 2}}, // RW_REGS
      {{h12, 0, 0, 0}, {h122, 1, -2, 2}} // RD_FIFO
      // ENC_TRAN not supported
    };

    // convenience macro
    inline bool checkFn(Function fn) {return (fn > NONE) && (fn < LASTFN) && PktParams[fn].req.hdr.len;}
  }
}

// to have all Modbus includes in one place
#include "modbus_request.hh"
#include "modbus_reply.hh"
#include "modbus_client.hh"

#endif // !_modbus_h
