///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _copy_hh
#define _copy_hh

#include "comm_error.hh"


namespace Comm
{
  // return i-th byte of n in ordering given by E
  // (i.e. for big-endian, the MSB is at index 0)
  template<typename T, Endianness E> inline uint8_t byteof(int i, T n)
  {
    if ((i < 0) || (i >= sizeof(T))) return 0;
    else
      switch (MachineEndian)
      {
      case LE:
	return reinterpret_cast<uint8_t *>(&n)[E == LE? i: sizeof(T)-i-1];
      case BE:
	return reinterpret_cast<uint8_t *>(&n)[E == BE? i: sizeof(T)-i-1];
      default:
	throw Error(GENERIC, "Modbus::byteof", "unsupported endianness");
      }
  }


  //// classes

  // a pathetic attempt at metaprogramming: appends bytes from item of type T 
  // to a buffer accessed by byte-copying function F with elementary copy
  // unit of S bytes (target buffer is big endian for now -- Modbus 
  // requirement);
  // S can be only 1 or 2, as Modbus doesn't support anything bigger than 
  // 2-byte words
  template<class T, int S, int I, class F>
  struct WordCopy: public WordCopy<T,S,(I > 0? I-1: 0),F>
  {
    void copy(T item, F& cf)
    {
      if (sizeof(T) < I) cf(0);
      else cf(byteof<T,LE>(I-1, item));
      WordCopy<T,S,(I > 0? I-1: 0),F>::copy(item, cf);
    }
  };

  // stop recursion on I
  template<class T, int S, class F> struct WordCopy<T,S,0,F>
  {
    void copy(T item, F& cf) { }
  };

  template<class T, int S, int I, class F>
  struct AppendItem: public AppendItem<T,S,(I-S > 0? I-S: 0),F>
  {
    void copy(T item, F& cf) {WordCopy<T,S,I,F>().copy(item, cf);}
  };

  // recursion end
  template<class T, int S, class F> struct AppendItem<T,S,0,F>
  {
    void copy(T item, F& cf) { }
  };

  // top level wrapper
  template<class T, int S, class F>
  struct Append: public AppendItem<T,S,(sizeof(T) > S? sizeof(T): S),F>
  {
    F& copyfn;
    
    Append(F& cf): copyfn(cf) { }
    inline void copy(T item) {AppendItem<T,S,(sizeof(T) > S? sizeof(T): S),F>::copy(item, copyfn);}
  };

  // byte copy functor
  template<class C> struct Copier
  {
    C *optr;
    void (C::*mptr)(uint8_t);

    Copier(C *op, void (C::*mp)(uint8_t)): optr(op), mptr(mp) { }
    void operator()(uint8_t item) {(optr->*mptr)(item);}
  };
}

#endif // !_copy_hh
