///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2008-2010 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <cstring>
#include <cstdlib>
#include <string>
#include <errno.h>
#include "client.hh"
#include "comm_error.hh"


namespace Comm
{
  // relocate unread data to the beginning of receive buffer
  void TCPClient::reloc()
  {
    if (start <= 0) return;
    memmove(recbuf, recbuf+start, (end-start));
    end -= start; start = 0;
  }

  // initialize receive buffer
  void TCPClient::initbuf()
  {
    bufsize = start = end = 0;
    recbuf = reinterpret_cast<char *>(malloc(AVG_RECV_BUF));
    if (!recbuf)
      throw Error(GENERIC, "Client::initbuf", "cannot allocate receiver buffer");
    bufsize = AVG_RECV_BUF;
  }

  // grow buffer if necessary to accommodate message of size sz
  void TCPClient::growbuf(int sz)
  {
    // sufficient capacity to appended to the current contents?
    if (sz < bufsize-end) return;

    // can it be appended to contents shifted to the beginning of buffer?
    if (sz < bufsize-(end-start))
    {
      reloc();
      return;
    }

    // have to reallocate...
    sz = (sz+AVG_RECV_BUF-1)/AVG_RECV_BUF;
    char *newp = reinterpret_cast<char *>(realloc(recbuf, sz));
    if (!newp)
      throw Error(GENERIC, "Client::growbuf", "resize of receive buffer failed");
    recbuf = newp; bufsize = sz;
    reloc();
  }

  // send buffer contents pointed to by ptr of length len
  void TCPClient::send(void *ptr, int len)
  {
    int remain = len;
    while (remain > 0)
    {
      int sent = ::send(sockid, ptr, len, 0);
      if (sent < 0)
	throw Error(GENERIC, "Client::send", "send to server failed");
      /*
	printf("sent: ");
	for (int i = 0; i < sent; i++)
	printf("%02x ", static_cast<uint8_t>(reinterpret_cast<char *>(ptr)[i]));
	printf("\n");
      */
      remain -= sent; ptr = reinterpret_cast<char *>(ptr)+sent;
    }
  }

  // receive len bytes, unless enough data are already available in receive buffer
  void TCPClient::recv(int len)
  {
    if (end-start >= len) return;

    growbuf(len);
    char *ptr = recbuf+end;
    int remain = len-(end-start);

    while (remain > 0)
    {
      int rcvd = ::recv(sockid, ptr, remain, 0);
      if (rcvd < 0)
	throw Error(TIMEOUT, "TCPClient::recv", "negative return code from recv()");
      remain -= rcvd; ptr += rcvd; end += rcvd;
    }
  }

  // re-synchronize stream (discard all bytes sitting in the kernel buffer)
  void TCPClient::resync()
  {
    int n;
    char buf[16];
    do {n = ::recv(sockid, buf, 16, MSG_DONTWAIT);}
    while (n > 0);
  }

  // connect to the server
  void TCPClient::connect(const char *host, int port)
  {
    sockid = socket(AF_INET, SOCK_STREAM, 0);
    if (sockid < 0)
      throw Error(GENERIC, "TCPClient::connect", "cannot create new socket");

    hostent *he = gethostbyname(host);
    if (!he)
    {
      std::string s = "server name lookup for ";
      s += host; s += " failed";
      throw Error(NOCONNECT, "TCPClient::connect", "host name lookup failed");
    }
  
    sockaddr_in sin;
    sin.sin_family = he->h_addrtype;
    sin.sin_port = htons(port);
    memcpy(&sin.sin_addr, he->h_addr, he->h_length);
    if (::connect(sockid, reinterpret_cast<sockaddr *>(&sin), sizeof(sin)) < 0)
    {
      std::string s = "cannot connect to ";
      s += host; s += ":"; s += port;
      throw Error(NOCONNECT, "TCPClient::connect", s.c_str());
    }

    if (flags & NET485_COMM_HEADER)
    { // no idea if this is needed with any other RS-485 bridges...
      static const char hdr[] = {0xff, 0xfb, 0x01, 0xff, 0xfb, 0x03};
  
      recv(sizeof(hdr));
      if (::strncmp(hdr, recbuf+start, sizeof(hdr)))
	throw Error(PROTOCOL, "TCPClient::connect", "failed to receive connection header from the server");
      start += sizeof(hdr);
    }
  }

  // set socket's receive timeout
  void TCPClient::setTimeout(long usec)
  {
    if (sockid < 0)
      throw Error(GENERIC, "TCPClient::setTimeout", "socket not open");
    timeval tv = {usec/1000000, usec%1000000}; 
    if (setsockopt(sockid, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv)))
      throw Error(GENERIC, "TCPClient::setTimeout", "cannot set receive timeout");
  }

  // append cnt bytes from external buffer to the receive buffer
  void TCPClient::get(void *buf, int cnt)
  {
    recv(cnt); memcpy(buf, recbuf+start, cnt); start += cnt;
  }
}
