///////////////////////////////////////////////////////////////////////////////
//
// Simple application launcher and meter access program
//
// Copyright (c) 2008-2010 Maciej Brodowicz
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying 
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
///////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstdlib>
#include <unistd.h>
#include <stdint.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <iostream>
#include <boost/thread.hpp>
#include <boost/program_options.hpp>
#include <boost/static_assert.hpp>
#include "modbus.hh"


// format to print timestamps
#define TIMEFMT "%8.3f"

namespace po = boost::program_options;
using namespace Comm;

// monitor error classes
enum AppErrNum
{
  OPTION = 0,
  INITIAL,
  RUNTIME
};

struct AppErrCat
{
  static char const *info(int t)
  {
    switch(t)
    {
    case OPTION:  return "[option processing error]";
    case INITIAL: return "[initialization error]";
    case RUNTIME: return "[runtime error]";
    default:      return "[invalid application error category]";
    }
  }
};

typedef ::Error<AppErrCat> AppError;

// static meter configuration
const int AC_STATION_BASE = 10;
const int DC_STATION_BASE = 20;
const int AC_PORT = 10001;
const int DC_PORT = 10001;
const char *AC_IP = "192.168.7.101";
const char *DC_IP = "192.168.7.102";
const long AC_TIMEOUT_USEC = 70000; // 70 ms
const long DC_TIMEOUT_USEC = 90000; // 90 ms
const int DCregnum = 516;    // first channel's register number
const double DCfact = 0.001; // scaling factor (raw reg. value -> amps)
const int DCchan = 4;        // total channels
// default output strem
FILE *OUTSTREAM = stdout;

// meter selectors
enum Mode
{
  MODE_NONE = 0,
  MODE_AC   = 1,
  MODE_DC   = 2,
  MODE_ALL  = MODE_AC | MODE_DC
};

//// global program parameters
// sampling period
double delay;
// sampling pad, index of accessed AC register in regtab
int pad, regi;
// output stream
FILE *ofile = OUTSTREAM;
// accessed meter group(s)
int active = MODE_NONE;
// print out DC measurements in Watts instead Amperes
bool dc_watts = true;
// nodes to monitor
std::vector<int> nodes;

// other config
boost::barrier *appsync; // global application start sync
double base; // initial value of RT clock converted to double
const int MAX_ID_LEN = 8;

//// convenience functions
// synchronized output
void safe_print(const char *fmt, ...)
{
  va_list ap;
  va_start(ap, fmt);

  static boost::mutex plock;
  plock.lock();
  vfprintf(ofile, fmt, ap); fprintf(ofile, "\n");
  plock.unlock();
}

// fractional part
inline double frac(double n) {return n-floor(n);}
// high-res time stamp
inline double timestamp()
{
  timespec ts;
  clock_gettime(CLOCK_REALTIME, &ts);
  return ts.tv_sec+1e-9*ts.tv_nsec-base;
}


// description of interesting AC meter (PR300) registers
struct ACreg
{
  const char *id, *unit;
  int reg;
  bool fp;
} regtab[] =
{
  {"active_power", "W",            0x14, true},
  {"reactive_power", "VA",         0x16, true},
  {"apparent_power", "VA",         0x18, true},
  {"active_energy", "kWh",         0x0, false},
  {"regenerative_energy", "kWh",   0x2, false},
  {"lead_reactive_energy", "kVAh", 0x4, false},
  {"lag_reactive_energy", "kVAh",  0x6, false},
  {"apparent_energy", "kVAh",      0x8, false},
  {"voltage", "V",                 0x1a, true},
  {"current", "A",                 0x20, true},
  {"power_factor", "",             0x26, true},
  {"frequency", "Hz",              0x28, true},
};


// base for printer objects
class Print
{
  std::string id_;

protected:
  std::string mkprfmt(char const *quant, char const *qfmt)
  {
    std::ostringstream s;
    s << id_ << ": " << TIMEFMT << " -> " << quant << ": " << qfmt;
    return s.str();
  }

public:
  Print(int node, char const *kind)
  {
    char nnum[8];
    snprintf(nnum, 8, "[%02d]", node);
    id_ = kind; id_ += nnum;
  }

  char const *id() {return id_.c_str();}
  void warn(double time, const char *msg)
  {
    safe_print("%s: " TIMEFMT " !! FAILURE: %s", id(), time, msg);
  }
};

// custom printer for AC meters
class ACprint: public Print
{
  std::string fmt;

public:
  static char const *kind() {return "AC";}

  ACprint(int node): Print(node, kind())
  {
    std::string lab, qf;
    for (int i = 0; regtab[regi].id[i]; i++)
      lab += (regtab[regi].id[i] == '_')? ' ': regtab[regi].id[i];
    qf = regtab[regi].fp? "%.2f ": "%u "; qf += regtab[regi].unit;
    fmt = mkprfmt(lab.c_str(), qf.c_str());
  }

  void display(double time, Modbus::Reply<Modbus::RTU> *r) const
  {
    uint16_t dbuf[2];

    // following IEC-1311
    // note that all reads of PR300 registers are 4 bytes long
    BOOST_STATIC_ASSERT(MachineEndian == BE || MachineEndian == LE);
    switch (MachineEndian)
    {
    // note that byte order in 16-bit datum already matches the machine order
    case BE:
      r->get(dbuf[1]);
      r->get(dbuf[0]);
      break;
    case LE:
      r->get(dbuf[0]);
      r->get(dbuf[1]);
      break;
    }
    if (regtab[regi].fp)
      safe_print(fmt.c_str(), time, *reinterpret_cast<float *>(dbuf));
    else
      safe_print(fmt.c_str(), time, *reinterpret_cast<uint32_t *>(dbuf));
  }
};

// custom printer for DC meters
class DCprint: public Print
{
  static double const rail_[];
  static char const *label_[];
  static char const *fmtW_[];
  static char const *fmtA_[];

  double scale_[DCchan];
  std::string prfmt_[DCchan], id_;

public:
  static char const *kind() {return "DC";}

  DCprint(int node): Print(node, kind())
  {
    for (int i = 0; i < DCchan; i++)
    {
      prfmt_[i] = mkprfmt(label_[i], dc_watts? fmtW_[i]: fmtA_[i]);
      scale_[i] = dc_watts? rail_[i]*DCfact: DCfact;
    }
  }

  void display(double time, Modbus::Reply<Modbus::RTU> *r)
  {
    uint32_t val;
    for (int i = 0; i < 4; i++)
    {
      uint16_t w;
      r->get(w); val = w;
      r->get(w); val |= static_cast<uint32_t>(w) << 16;
      safe_print(prfmt_[i].c_str(), time, scale_[i]*val);
    }
  }
};

// DC display related constants
double const DCprint::rail_[] = {3.3, 5, 12, 12};
char const *DCprint::label_[] = {"3.3V rail", "5V rail", "12V rail1", "12V rail2"};
char const *DCprint::fmtW_[] = {"%5.2f W", " %5.2f W", "%5.2f W", "%6.2f W"};
char const *DCprint::fmtA_[] = {"%5.3f A", " %5.3f A", "%5.3f A", "%6.3f A"};


// single meter representation
template<Mode M, typename PR>
class Meter: public PR
{
  BOOST_STATIC_ASSERT(M == MODE_AC || M == MODE_DC);

  Modbus::Request<Modbus::RTU> request_;
  char *id_;

  void initreq(int regbase);

public:
  Meter(int node, int station, int regbase):
    PR(node), request_(station, Modbus::RD_HREGS)
  {
    initreq(regbase);
  }

  void sample(Modbus::Client<Modbus::RTU>&);
};

typedef Meter<MODE_AC, ACprint> ACmeter;
typedef Meter<MODE_DC, DCprint> DCmeter;

// callable for threads performing sampling
template<Mode M, typename P> class Sthread
{
  BOOST_STATIC_ASSERT(M == MODE_AC || M == MODE_DC);

  Modbus::Client<Modbus::RTU> client_;
  std::vector<Meter<M, P> > meters_;
  boost::mutex timelock_;
  long now_, end_;

  inline long ticks()
  {
    timelock_.lock();
    long n = now_;
    timelock_.unlock();
    return n;
  }

public:
  Sthread(std::vector<int> const& nlist, const char *server, int port);

  void operator()()
  {
    try
    {
      safe_print("%s: starting monitor thread", meters_[0].id());
      while (true)
      {
	// test for completion
	long n = ticks();
	if (n > end_) return;
	// check if it's time to start the app
	if (n == pad) appsync->wait();

	// obtain samples from all selected meters on this bridge
	for (int i = 0; i < meters_.size(); i++) meters_[i].sample(client_);

	// sleep until the next sample time
	double dnow = timestamp();
	double next = base+delay*(static_cast<long>(dnow/delay)+1);
	timespec ts = {static_cast<time_t>(floor(next)), static_cast<long>(frac(next)*1e9)};
	while (clock_nanosleep(CLOCK_REALTIME, TIMER_ABSTIME, &ts, 0)) ;
	timelock_.lock();
	now_++;
	timelock_.unlock();
      }
      safe_print("%s: exiting monitor thread", meters_[0].id());
    }
    catch (std::exception& e)
    {
      safe_print("Error: %s in %s monitor thread", e.what(), meters_[0].id());
    }
  }

  inline void ending(int pad)
  { // calculate last tick number
    timelock_.lock();
    end_ = pad+now_+1;
    timelock_.unlock();
  }
};

template<>
Sthread<MODE_AC, ACprint>::Sthread(std::vector<int> const& nlist, const char *server, int port):
  now_(0), end_(1L<<30),
  client_(server, port, Modbus::PR300_NULL_FN, NET485_COMM_HEADER)
{
  client_.setTimeout(AC_TIMEOUT_USEC);
  for (int i = 0; i < nlist.size(); i++)
    meters_.push_back(ACmeter(nodes[i], AC_STATION_BASE+nlist[i], regtab[regi].reg));
}

template<>
Sthread<MODE_DC, DCprint>::Sthread(std::vector<int> const& nlist, const char *server, int port):
  now_(0), end_(1L<<30),
  client_(server, port, Modbus::PR300_NULL_FN, NET485_COMM_HEADER)
{
  client_.setTimeout(DC_TIMEOUT_USEC);
  for (int i = 0; i < nlist.size(); i++)
    meters_.push_back(DCmeter(nodes[i], DC_STATION_BASE+nlist[i], DCregnum));
}

template<> void ACmeter::initreq(int reg)
{
  request_.put(reg); request_.put(2); // uint32 or float => 2x16 bit words
}

template<> void DCmeter::initreq(int reg)
{
  request_.put(reg); request_.put(8); // 4x32bit regs => 8x16bit words
}

template<Mode M, typename P>
void Meter<M, P>::sample(Modbus::Client<Modbus::RTU>& cl)
{
  Modbus::Reply<Modbus::RTU> *resp;

  // send request, receive reply, get the corresponding timestamps
  double st = timestamp();
  bool valid = true;
  try
  {
    cl.send(&request_);
    resp = cl.recv();
  }
  catch (Comm::Error& err)
  {
    std::string msg;
    switch (err.errnum())
    {
    case CHECKSUM:
    case TIMEOUT:
    case CORRUPT:
      valid = false;
      msg = err.category(); msg += " "; msg += err.info();
      P::warn(timestamp(), msg.c_str());
      break;
    default: // all others handled in wrapper
      throw err;
    }
  }

  if (valid)
  { // correctly formed reply was received
    double rt = timestamp();
    if (resp->failed()) // request execution failed
      P::warn((st+rt)/2, Modbus::ExcInfo[resp->ecode]);
    else P::display((st+rt)/2, resp);
  }
}

// find unique match for a label (could be partial string)
int find_match(const std::string& lab)
{
  int i, n;
  std::string s;
  // convert space to underscores, make everything lowercase
  for (i = 0; lab[i]; i++) s += (lab[i] == ' ')? '_': tolower(lab[i]);

  const int len = sizeof(regtab)/sizeof(regtab[0]);
  for (n = -1, i = 0; i < len; i++)
    if (s == std::string(regtab[i].id).substr(0, strlen(regtab[i].id)))
    {
      if (n >= 0)
	throw AppError(OPTION, "find_match", "ambiguous register name: %s", s.c_str());
      n = i;
    }
  if (n < 0)
    throw AppError(OPTION, "find_match", "invalid register name: %s", s.c_str());
  return n;
}

// parse command line
std::string cmd_options(int ac, char *av[])
{
  int msdelay;

  std::string head("\nUsage: ");
  head += av[0];
  head += " [<option>...] <application> [<application_argument>...]\n";
  head += "\nSupported options";

  std::string app, rname(regtab[0].id), fname, nodstr;
  std::vector<std::string> cmd;
  // named options
  po::options_description desc(head);
  desc.add_options()
    ("current,c", "output current values instead of power for DC meters")
    ("nodes,n",
     po::value<std::vector<int> >(&nodes)->multitoken(),
     "list of monitored node numbers")
    ("output,o",
     po::value<std::string>(&fname),
     "dump measurements to a file")
    ("pad,p",
     po::value<int>(&pad)->default_value(5),
     "timing pad around application start and termination, in samples")
    ("register,r",
     po::value<std::string>(&rname)->default_value(regtab[0].id),
     "name of the register to be read from the AC meter")
    ("period,s",
     po::value<int>(&msdelay)->default_value(500),
     "sampling period, in milliseconds")
    ("ac,a", "monitor only AC meters")
    ("dc,d", "monitor only DC meters")
    ("help,h", "prints this message")
  ;
  // hidden options
  po::options_description hid("hidden");
  hid.add_options()
    ("command",
     po::value<std::vector<std::string> >(&cmd)->multitoken(),
     "command to start test application")
  ;

  // positional options: application command line
  po::positional_options_description pos;
  pos.add("command", -1);

  // parse command line
  po::options_description allopts;
  allopts.add(desc).add(hid);
  po::variables_map vm;
  po::store(po::command_line_parser(ac, av).options(allopts).positional(pos).run(), vm);
  po::notify(vm);

  if (vm.count("help"))
  {
    std::cout << desc << "\n"; exit(0);
  }
  if (!vm.count("nodes"))
    throw AppError(OPTION, "cmd_options", "missing the list of nodes to monitor");
  if (vm.count("ac")) active |= MODE_AC;
  if (vm.count("dc")) active |= MODE_DC;
  // default is to monitor everything
  if (active == MODE_NONE) active = MODE_ALL;
  // format and multipliers for DC measurements
  if (vm.count("current")) dc_watts = false;
  // output stream
  if (!fname.empty() && !(ofile = fopen(fname.c_str(), "w")))
    throw AppError(OPTION, "cmd_options", "failed to open output file for writing");
  // find monitored AC register by its id
  if (vm.count("register")) regi = find_match(rname);
  // express delay in seconds
  delay = msdelay/1000.0;

  // assemble application command
  if (!vm.count("command"))
    throw AppError(OPTION, "cmd_options", "missing application name and arguments");
  else
  {
    int i;
    for (i = 0; i < vm["command"].as<std::vector<std::string> >().size(); i++)
    {
      if (i) app += ' ';
      app += vm["command"].as<std::vector<std::string> >()[i];
    }
  }
  return app;
}

// set timing
void inittime()
{
  if (((active & MODE_AC) && delay < AC_TIMEOUT_USEC/1000000.0) ||
      ((active & MODE_DC) && delay < DC_TIMEOUT_USEC/1000000.0))
    std::cerr << "Warning: sampling period smaller than communication timeout\n";
  timespec ts, tsbase;
  if (clock_getres(CLOCK_REALTIME, &ts))
    throw AppError(INITIAL, "run", "CLOCK_REALTIME is not available");
  if ((ts.tv_sec > 0) || (ts.tv_nsec > 10000000))
    throw AppError(INITIAL, "run", "CLOCK_REALTIME has resolution worse than 10ms");
  clock_gettime(CLOCK_REALTIME, &tsbase);
  base = tsbase.tv_sec+tsbase.tv_nsec*1e-9;
}

// run application while sampling the meters
void run(std::string app)
{
  // various intializations
  inittime();

  // initialize the barrier and start sampling thread(s)
  Sthread<MODE_AC, ACprint> *ac_sampler = 0;
  Sthread<MODE_DC, DCprint> *dc_sampler = 0;
  int numthr = 1+(active & MODE_AC? 1: 0)+(active & MODE_DC? 1: 0);
  boost::thread *ac_thr = 0, *dc_thr = 0;
  appsync = new boost::barrier(numthr);

  if (active & MODE_AC)
  {
    ac_sampler = new Sthread<MODE_AC, ACprint>(nodes, AC_IP, AC_PORT);
    ac_thr = new boost::thread(boost::ref(*ac_sampler));
  }
  if (active & MODE_DC)
  {
    dc_sampler = new Sthread<MODE_DC, DCprint>(nodes, DC_IP, DC_PORT);
    dc_thr = new boost::thread(boost::ref(*dc_sampler));
  }

  // invoke remote application
  appsync->wait();
  safe_print("******: " TIMEFMT " ** Starting: %s", timestamp(), app.c_str());
  system(app.c_str());

  // application execution finished
  safe_print("******: " TIMEFMT " ** Application exited", timestamp());

  // compute termination timestamps for sampling threads
  if (ac_sampler) ac_sampler->ending(pad);
  if (dc_sampler) dc_sampler->ending(pad);

  // wait for the thread(s) to terminate
  if (ac_thr)
  {
    ac_thr->join(); delete ac_thr; delete ac_sampler;
  }
  if (dc_thr)
  {
    dc_thr->join(); delete dc_thr; delete dc_sampler;
  }
}

int main(int argc, char *argv[])
{
  int rc = 0;

  try
  {
    run(cmd_options(argc, argv));
  }
  catch (std::exception& e)
  {
    std::cerr << "Error " << e.what() << " (in main thread)\n";
    rc = 1;
  }

  if (ofile != OUTSTREAM) fclose(ofile);
  return rc;
}
