#include <npp.h>

#include <ImagesCPU.h>
#include <ImagesNPP.h>
#include <ImageIO.h>
#include <Exceptions.h>

#include <string>
#include <iostream>


int main(int argc, char* argv[])
{
    try
    {
        // if more than one command line arg, use the first arg as the filename,
        // otherwise assume the filename included with the sample
        std::string fileName = "../../data/Lena.pgm";

        if (argc >= 2)
            fileName = argv[1];

        std::string dstFileName = fileName;
        
        std::string::size_type dot = dstFileName.rfind('.');
        if (dot != std::string::npos) dstFileName = dstFileName.substr(0, dot);
        dstFileName += "_histEqualization.pgm";

        if (argc >= 3)
            dstFileName = argv[2];

        npp::ImageCPU_8u_C1 oHostSrc;
        npp::loadImage(fileName, oHostSrc);
        npp::ImageNPP_8u_C1 oDeviceSrc(oHostSrc);

        //
        // allocate arrays for histogram and levels
        //

        const int binCount = 256;
        const int levelCount = binCount + 1; // levels array has one more element

        Npp32s * histDevice = 0;
        Npp32s * levelsDevice = 0;
            
        NPP_CHECK_CUDA(cudaMalloc((void **)&histDevice,   binCount   * sizeof(Npp32s)));
        NPP_CHECK_CUDA(cudaMalloc((void **)&levelsDevice, levelCount * sizeof(Npp32s)));

        //
        // compute histogram
        //

        NppiSize oSizeROI = {oDeviceSrc.width(), oDeviceSrc.height()}; // full image
                // create device scratch buffer for nppiHistogram
        int nDeviceBufferSize;
        nppiHistogramEvenGetBufferSize_8u_C1R(oSizeROI, levelCount ,&nDeviceBufferSize);
        Npp8u * pDeviceBuffer;
        NPP_CHECK_CUDA(cudaMalloc((void **)&pDeviceBuffer, nDeviceBufferSize));
        
                // compute levels values on host
        Npp32s levelsHost[levelCount];
        NPP_CHECK_NPP(nppiEvenLevelsHost_32s(levelsHost, levelCount, 0, binCount));
                // compute the histogram
        NPP_CHECK_NPP(nppiHistogramEven_8u_C1R(oDeviceSrc.data(), oDeviceSrc.pitch(), oSizeROI, 
                                               histDevice, levelCount, 0, binCount, 
                                               pDeviceBuffer));
                // copy histogram and levels to host memory
        Npp32s histHost[binCount];
        NPP_CHECK_CUDA(cudaMemcpy(histHost, histDevice, binCount * sizeof(Npp32s), cudaMemcpyDeviceToHost));

        Npp32s  lutHost[binCount + 1];

                // fill LUT
        {
            Npp32s * pHostHistogram = histHost;
            Npp32s totalSum = 0;
            for (; pHostHistogram < histHost + binCount; ++pHostHistogram)
                totalSum += *pHostHistogram;

            NPP_ASSERT(totalSum == oSizeROI.width * oSizeROI.height);

            if (totalSum == 0) 
                totalSum = 1;
            float multiplier = 1.0f / float(totalSum) * 0xFF;

            Npp32s runningSum = 0;
            Npp32s * pLookupTable = lutHost;
            for (pHostHistogram = histHost; pHostHistogram < histHost + binCount; ++pHostHistogram)
            {
                *pLookupTable = (Npp32s)(runningSum * multiplier + 0.5f);
                pLookupTable++;
                runningSum += *pHostHistogram;
            }

            lutHost[binCount] = 0xFF; // last element is always 1
        }

        //
        // apply LUT transformation to the image
        //
                // Create a device image for the result.
        npp::ImageNPP_8u_C1 oDeviceDst(oDeviceSrc.size());
        NPP_CHECK_NPP(nppiLUT_Linear_8u_C1R(oDeviceSrc.data(), oDeviceSrc.pitch(), 
                                            oDeviceDst.data(), oDeviceDst.pitch(), 
                                            oSizeROI, 
                                            lutHost, // value and level arrays are in host memory
                                            levelsHost, 
                                            binCount+1));

                // copy the result image back into the storage that contained the 
                // input image
        npp::ImageCPU_8u_C1 oHostDst(oDeviceDst.size());
        oDeviceDst.copyTo(oHostDst.data(), oHostDst.pitch());

                // save the result
        npp::saveImage(dstFileName.c_str(), oHostDst);

        std::cout << "Saved image file " << dstFileName << std::endl;
    }
    catch (npp::Exception & rException)
    {
        std::cerr << "Program error! The following exception occurred: \n";
        std::cerr << rException << std::endl;
        std::cerr << "Aborting." << std::endl;
        
        return -1;
    }
    catch (...)
    {
        std::cerr << "Program error! An unknow type of exception occurred. \n";
        std::cerr << "Aborting." << std::endl;
        
        return -1;
    }
    
    return 0;
}

