/*
* Copyright 1993-2007 NVIDIA Corporation.  All rights reserved.
*
* NOTICE TO USER:
*
* This source code is subject to NVIDIA ownership rights under U.S. and
* international Copyright laws.
*
* NVIDIA MAKES NO REPRESENTATION ABOUT THE SUITABILITY OF THIS SOURCE
* CODE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR
* IMPLIED WARRANTY OF ANY KIND.  NVIDIA DISCLAIMS ALL WARRANTIES WITH
* REGARD TO THIS SOURCE CODE, INCLUDING ALL IMPLIED WARRANTIES OF
* MERCHANTABILITY, NONINFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE.
* IN NO EVENT SHALL NVIDIA BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL,
* OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
* OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
* OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
* OR PERFORMANCE OF THIS SOURCE CODE.
*
* U.S. Government End Users.  This source code is a "commercial item" as
* that term is defined at 48 C.F.R. 2.101 (OCT 1995), consisting  of
* "commercial computer software" and "commercial computer software
* documentation" as such terms are used in 48 C.F.R. 12.212 (SEPT 1995)
* and is provided to the U.S. Government only as a commercial end item.
* Consistent with 48 C.F.R.12.212 and 48 C.F.R. 227.7202-1 through
* 227.7202-4 (JUNE 1995), all U.S. Government End Users acquire the
* source code with only those rights set forth herein.
*/

#include "FreeImage.h"

#include "Exceptions.h"

#include <cuda_runtime.h>
#include <npp.h>
#include <iostream>


// Error handler for FreeImage library. 
//  In case this handler is invoked, it throws an NPP exception.
void 
FreeImageErrorHandler(FREE_IMAGE_FORMAT oFif, const char * zMessage) 
{
    throw npp::Exception(zMessage);
}

std::ostream &
operator <<(std::ostream & rOutputStream, const FIBITMAP & rBitmap)
{
    unsigned int nImageWidth    = FreeImage_GetWidth(const_cast<FIBITMAP *>(&rBitmap));
    unsigned int nImageHeight   = FreeImage_GetHeight(const_cast<FIBITMAP *>(&rBitmap));
    unsigned int nPitch         = FreeImage_GetPitch(const_cast<FIBITMAP *>(&rBitmap));
    unsigned int nBPP           = FreeImage_GetBPP(const_cast<FIBITMAP *>(&rBitmap));
    
    FREE_IMAGE_COLOR_TYPE eType = FreeImage_GetColorType(const_cast<FIBITMAP *>(&rBitmap));
    BITMAPINFO * pInfo          = FreeImage_GetInfo(const_cast<FIBITMAP *>(&rBitmap));
    
    rOutputStream << "Size  (" << FreeImage_GetWidth(const_cast<FIBITMAP *>(&rBitmap)) << ", " 
                               << FreeImage_GetHeight(const_cast<FIBITMAP *>(&rBitmap)) << ")\n";
    rOutputStream << "Pitch "  << FreeImage_GetPitch(const_cast<FIBITMAP *>(&rBitmap)) << "\n";
    rOutputStream << "Type  ";
    switch (eType)
    {
    case FIC_MINISWHITE:
        rOutputStream << "FIC_MINISWHITE\n";
    break;
    case FIC_MINISBLACK:
        rOutputStream << "FIC_MINISBLACK\n";
    break;
    case FIC_RGB:
        rOutputStream << "FIC_RGB\n";
    break;
    case FIC_PALETTE:
        rOutputStream << "FIC_PALETTE\n";
    break;
    case FIC_RGBALPHA:
        rOutputStream << "FIC_RGBALPHA\n";
    break;
    case FIC_CMYK:
        rOutputStream << "FIC_CMYK\n";
    break;
    default:
        rOutputStream << "Unknown pixel format.\n";
    }
    rOutputStream << "BPP   " << nBPP << std::endl;
    
    return rOutputStream;
}

int 
main(int argc, char *argv[]) 
{
    try
    {
	            // set your own FreeImage error handler
	    FreeImage_SetOutputMessage(FreeImageErrorHandler);

        // if more than one command line arg, use the first arg as the filename,
        // otherwise assume the filename included with the sample
        std::string sFilename = "../../data/Lena.pgm";

        if (argc >= 2)
            sFilename = argv[1];

        std::string sResultFilename = sFilename;
        
        std::string::size_type dot = sResultFilename.rfind('.');
        if (dot != std::string::npos) sResultFilename = sResultFilename.substr(0, dot);
        sResultFilename += "_boxFilterFII.pgm";

        if (argc >= 3)
            sResultFilename = argv[2];

        FREE_IMAGE_FORMAT eFormat = FreeImage_GetFileType(sFilename.c_str());
                // no signature? try to guess the file format from the file extension
        if (eFormat == FIF_UNKNOWN)
            eFormat = FreeImage_GetFIFFromFilename(sFilename.c_str());
        NPP_ASSERT(eFormat != FIF_UNKNOWN);
                // check that the plugin has reading capabilities ...
        FIBITMAP * pBitmap;
        if (FreeImage_FIFSupportsReading(eFormat)) 
            pBitmap = FreeImage_Load(eFormat, sFilename.c_str());
        NPP_ASSERT(pBitmap != 0);
                // Dump the bitmap information to the console
        std::cout << (*pBitmap) << std::endl;
                // make sure this is an 8-bit single channel image
        NPP_ASSERT(FreeImage_GetColorType(pBitmap) == FIC_MINISBLACK);
        NPP_ASSERT(FreeImage_GetBPP(pBitmap) == 8);
        
        unsigned int nImageWidth  = FreeImage_GetWidth(pBitmap);
        unsigned int nImageHeight = FreeImage_GetHeight(pBitmap);
        unsigned int nSrcPitch    = FreeImage_GetPitch(pBitmap);
        unsigned char * pSrcData  = FreeImage_GetBits(pBitmap);
        
        int nSrcPitchCUDA;
        Npp8u * pSrcImageCUDA = nppiMalloc_8u_C1(nImageWidth, nImageHeight, &nSrcPitchCUDA);
        NPP_ASSERT_NOT_NULL(pSrcImageCUDA);
                // copy image loaded via FreeImage to into CUDA device memory, i.e.
                // transfer the image-data up to the GPU's video-memory
        NPP_CHECK_CUDA(cudaMemcpy2D(pSrcImageCUDA, nSrcPitchCUDA, pSrcData, nSrcPitch, 
                                nImageWidth, nImageHeight, cudaMemcpyHostToDevice));
                // define size of the box filter
        const NppiSize  oMaskSize   = {7, 7};
        const NppiPoint oMaskAchnor = {0, 0};
                // compute maximal result image size
        const NppiSize  oSizeROI = {nImageWidth  - (oMaskSize.width - 1),
                                     nImageHeight - (oMaskSize.height - 1)};
                // allocate result image memory
        int nDstPitchCUDA; 
        Npp8u * pDstImageCUDA = nppiMalloc_8u_C1(oSizeROI.width, oSizeROI.height, &nDstPitchCUDA);
        NPP_ASSERT_NOT_NULL(pDstImageCUDA);
        NPP_CHECK_NPP(nppiFilterBox_8u_C1R(pSrcImageCUDA, nSrcPitchCUDA, pDstImageCUDA, nDstPitchCUDA, 
                                         oSizeROI, oMaskSize, oMaskAchnor));
                // create the result image storage using FreeImage so we can easily 
                // save
        FIBITMAP * pResultBitmap = FreeImage_Allocate(oSizeROI.width, oSizeROI.height, 8 /* bits per pixel */);
        NPP_ASSERT_NOT_NULL(pResultBitmap);
        unsigned int nResultPitch   = FreeImage_GetPitch(pResultBitmap);
        unsigned char * pResultData = FreeImage_GetBits(pResultBitmap);

        NPP_CHECK_CUDA(cudaMemcpy2D(pResultData, nResultPitch, pDstImageCUDA, nDstPitchCUDA,
                                oSizeROI.width, oSizeROI.height, cudaMemcpyDeviceToHost));
                // now save the result image
        bool bSuccess;
        bSuccess = FreeImage_Save(FIF_PGM, pResultBitmap, sResultFilename.c_str(), 0) == TRUE;
        NPP_ASSERT_MSG(bSuccess, "Failed to save result image.");
    }
    catch (npp::Exception & rException)
    {
        std::cerr << "Program error! The following exception occurred: \n";
        std::cerr << rException << std::endl;
        std::cerr << "Aborting." << std::endl;
        
        return -1;
    }
    catch (...)
    {
        std::cerr << "Program error! An unknow type of exception occurred. \n";
        std::cerr << "Aborting." << std::endl;
        
        return -1;
    }
    
    return 0;
}
