#include <npp.h>

#include <ImagesCPU.h>
#include <ImagesNPP.h>
#include <ImageIO.h>
#include <Exceptions.h>

#include <iostream>


int main(int argc, char* argv[])
{
    try
    {
        // if more than one command line arg, use the first arg as the filename,
        // otherwise assume the filename included with the sample
        std::string sFilename = "../../data/Lena.pgm";

        if (argc >= 2)
            sFilename = argv[1];

        std::string sResultFilename = sFilename;
        
        std::string::size_type dot = sResultFilename.rfind('.');
        if (dot != std::string::npos) sResultFilename = sResultFilename.substr(0, dot);
        sResultFilename += "_boxFilter.pgm";

        if (argc >= 3)
            sResultFilename = argv[2];

                // declare a host image object for an 8-bit grayscale image
        npp::ImageCPU_8u_C1 oHostSrc;
                // load gray-scale image from disk
        npp::loadImage(sFilename, oHostSrc);
                // declara a device image and copy construct from the host image,
                // i.e. upload host to device
        npp::ImageNPP_8u_C1 oDeviceSrc(oHostSrc);
                
                // create struct with box-filter mask size
        NppiSize oMaskSize = {5, 5};
                // create struct with ROI size given the current mask
        NppiSize oSizeROI = {oDeviceSrc.width() - oMaskSize.width + 1, oDeviceSrc.height() - oMaskSize.height + 1};
                // allocate device image of appropriatedly reduced size
        npp::ImageNPP_8u_C1 oDeviceDst(oSizeROI.width, oSizeROI.height);
                // set anchor point inside the mask to (0, 0)
        NppiPoint oAnchor = {0, 0};
                // run box filter
        NppStatus eStatusNPP;
        eStatusNPP = nppiFilterBox_8u_C1R(oDeviceSrc.data(), oDeviceSrc.pitch(), 
                                          oDeviceDst.data(), oDeviceDst.pitch(), 
                                          oSizeROI, oMaskSize, oAnchor);
        NPP_ASSERT(NPP_NO_ERROR == eStatusNPP);
                // declare a host image for the result
        npp::ImageCPU_8u_C1 oHostDst(oDeviceDst.size());
                // and copy the device result data into it
        oDeviceDst.copyTo(oHostDst.data(), oHostDst.pitch());
        
        saveImage(sResultFilename, oHostDst);
        std::cout << "Saved image: " << sResultFilename << std::endl;
    }
    catch (npp::Exception & rException)
    {
        std::cerr << "Program error! The following exception occurred: \n";
        std::cerr << rException << std::endl;
        std::cerr << "Aborting." << std::endl;
        
        return -1;
    }
    catch (...)
    {
        std::cerr << "Program error! An unknow type of exception occurred. \n";
        std::cerr << "Aborting." << std::endl;
        
        return -1;
    }
    
    return 0;
}