/*
* Copyright 1993-2010 NVIDIA Corporation.  All rights reserved.
*
* NOTICE TO USER:   
*
* This source code is subject to NVIDIA ownership rights under U.S. and 
* international Copyright laws.  
*
* NVIDIA MAKES NO REPRESENTATION ABOUT THE SUITABILITY OF THIS SOURCE 
* CODE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR 
* IMPLIED WARRANTY OF ANY KIND.  NVIDIA DISCLAIMS ALL WARRANTIES WITH 
* REGARD TO THIS SOURCE CODE, INCLUDING ALL IMPLIED WARRANTIES OF 
* MERCHANTABILITY, NONINFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE.   
* IN NO EVENT SHALL NVIDIA BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL, 
* OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS 
* OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE 
* OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE 
* OR PERFORMANCE OF THIS SOURCE CODE.  
*
* U.S. Government End Users.  This source code is a "commercial item" as 
* that term is defined at 48 C.F.R. 2.101 (OCT 1995), consisting  of 
* "commercial computer software" and "commercial computer software 
* documentation" as such terms are used in 48 C.F.R. 12.212 (SEPT 1995) 
* and is provided to the U.S. Government only as a commercial end item.  
* Consistent with 48 C.F.R.12.212 and 48 C.F.R. 227.7202-1 through 
* 227.7202-4 (JUNE 1995), all U.S. Government End Users acquire the 
* source code with only those rights set forth herein.
*/
#ifndef NV_NPPS_H
#define NV_NPPS_H
 
/**
 * \file npps.h
 * NPP Signal Processing Functionality.
 */
 
 #include "nppdefs.h"


#ifdef __cplusplus
extern "C" {
#endif

/** \defgroup npps NPP Signal Processing
 * @{
 */


/** @name Memory Allocation
 * Signal-allocator methods for allocating 1D arrays of data in device memory.
 * All allocators have size parameters to specify the size of the signal (1D array)
 * being allocated.
 *
 * The allocator methods return a pointer to the newly allocated memory of appropriate
 * type. If device-memory allocation is not possible due to resource constaints
 * the allocators return 0 (i.e. NULL pointer). 
 *
 * All signal allocators allocate memory aligned such that it is  beneficial to the 
 * performance of the majority of the signal-processing primitives. 
 * It is no mandatory however to use these allocators. Any valid
 * CUDA device-memory pointers can be passed to NPP primitives. 
 */
//@{

/**
 * 8-bit unsigned signal allocator.
 * \param nSize Number of unsigned chars in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp8u * 
nppsMalloc_8u(int nSize);

/**
 * 16-bit unsigned signal allocator.
 * \param nSize Number of unsigned shorts in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp16u * 
nppsMalloc_16u(int nSize);

/**
 * 16-bit signal allocator.
 * \param nSize Number of shorts in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp16s * 
nppsMalloc_16s(int nSize);

/**
 * 16-bit complex-value signal allocator.
 * \param nSize Number of 16-bit complex numbers in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp16sc * 
nppsMalloc_16sc(int nSize);

/**
 * 32-bit unsigned signal allocator.
 * \param nSize Number of unsigned ints in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp32u * 
nppsMalloc_32u(int nSize);

/**
 * 32-bit integer signal allocator.
 * \param nSize Number of ints in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp32s * 
nppsMalloc_32s(int nSize);

/**
 * 32-bit complex integer signal allocator.
 * \param nSize Number of complex integner values in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp32sc * 
nppsMalloc_32sc(int nSize);

/**
 * 32-bit float signal allocator.
 * \param nSize Number of floats in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp32f * 
nppsMalloc_32f(int nSize);

/**
 * 32-bit complex float signal allocator.
 * \param nSize Number of complex float values in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp32fc * 
nppsMalloc_32fc(int nSize);

/**
 * 64-bit long integer signal allocator.
 * \param nSize Number of long ints in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp64s * 
nppsMalloc_64s(int nSize);

/**
 * 64-bit complex long integer signal allocator.
 * \param nSize Number of complex long int values in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp64sc * 
nppsMalloc_64sc(int nSize);

/**
 * 64-bit float (double) signal allocator.
 * \param nSize Number of doubles in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp64f * 
nppsMalloc_64f(int nSize);

/**
 * 64-bit complex complex signal allocator.
 * \param nSize Number of complex double valuess in the new signal.
 * \return A pointer to the new signal. 0 (NULL-pointer) indicates
 *         that an error occurred during allocation.
 */
Npp64fc * 
nppsMalloc_64fc(int nSize);

/**
 * Free method for any 2D allocated memory.
 * This method should be used to free memory allocated with any of the
 * nppiMalloc_<modifier> methods.
 * \param pValues A pointer to memory allocated using nppiMalloc_<modifier>.
 */
void nppsFree(void * pValues);

// end of Memory Allocaiton
//@}

/** @name Set 
 * Set methods for 1D vectors of various types. The copy methods operate on vector data given
 * as a pointer to the underlying data-type (e.g. 8-bit vectors would
 * be passed as pointers to Npp8u type) and length of the vectors, i.e. the number of items.
 */
//@{

/** 
 * 8-bit unsigned char, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_8u(Npp8u nValue, Npp8u * pDst, int nLength);

/** 
 * 16-bit integer, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_16s(Npp16s nValue, Npp16s * pDst, int nLength);

/** 
 * 16-bit integer complex, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_16sc(Npp16sc nValue, Npp16sc * pDst, int nLength);

/** 
 * 32-bit integer, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_32s(Npp32s nValue, Npp32s * pDst, int nLength);

/** 
 * 32-bit integer complex, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_32sc(Npp32sc nValue, Npp32sc * pDst, int nLength);

/** 
 * 32-bit float, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_32f(Npp32f nValue, Npp32f * pDst, int nLength);

/** 
 * 32-bit float complex, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_32fc(Npp32fc nValue, Npp32fc * pDst, int nLength);

/** 
 * 64-bit long long integer, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_64s(Npp64s nValue, Npp64s * pDst, int nLength);

/** 
 * 64-bit long long integer complex, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_64sc(Npp64sc nValue, Npp64sc * pDst, int nLength);

/** 
 * 64-bit double, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_64f(Npp64f nValue, Npp64f * pDst, int nLength);

/** 
 * 64-bit double complex, vector set method.
 * \param nValue Value used to initialize the vector pDst.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSet_64fc(Npp64fc nValue, Npp64fc * pDst, int nLength);

// end of Signal Set
//@}

/** @name Zero
 * Set signals to zero.
 */
//@{

/** 
 * 8-bit unsigned char, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_8u(Npp8u * pDst, int nLength);

/** 
 * 16-bit integer, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_16s(Npp16s * pDst, int nLength);

/** 
 * 16-bit integer complex, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_16sc(Npp16sc * pDst, int nLength);

/** 
 * 32-bit integer, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_32s(Npp32s * pDst, int nLength);

/** 
 * 32-bit integer complex, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_32sc(Npp32sc * pDst, int nLength);

/** 
 * 32-bit float, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_32f(Npp32f * pDst, int nLength);

/** 
 * 32-bit float complex, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_32fc(Npp32fc * pDst, int nLength);

/** 
 * 64-bit long long integer, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_64s(Npp64s * pDst, int nLength);

/** 
 * 64-bit long long integer complex, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_64sc(Npp64sc * pDst, int nLength);

/** 
 * 64-bit double, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_64f(Npp64f * pDst, int nLength);

/** 
 * 64-bit double complex, vector zero method.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsZero_64fc(Npp64fc * pDst, int nLength);

// end of Zero
//@}

/** @name Copy
 * Copy methods for various type signals. Copy methods operate on
 * signal data given as a pointer to the underlying data-type (e.g. 8-bit
 * vectors would be passed as pointers to Npp8u type) and length of the
 * vectors, i.e. the number of items. 
 */
//@{

/** 
 * 8-bit unsigned char, vector copy method
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_8u(const Npp8u * pSrc, Npp8u * pDst, int len);

/** 
 * 16-bit signed short, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_16s(const Npp16s * pSrc, Npp16s * pDst, int len);

/** 
 * 32-bit signed integer, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param nLength \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_32s(const Npp32s * pSrc, Npp32s * pDst, int nLength);

/** 
 * 32-bit float, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_32f(const Npp32f * pSrc, Npp32f * pDst, int len);

/** 
 * 64-bit signed integer, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_64s(const Npp64s * pSrc, Npp64s * pDst, int len);

/** 
 * 16-bit complex short, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_16sc(const Npp16sc * pSrc, Npp16sc * pDst, int len);

/** 
 * 32-bit complex signed integer, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_32sc(const Npp32sc * pSrc, Npp32sc * pDst, int len);

/** 
 * 32-bit complex float, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_32fc(const Npp32fc * pSrc, Npp32fc * pDst, int len);

/** 
 * 64-bit complex signed integer, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_64sc(const Npp64sc * pSrc, Npp64sc * pDst, int len);

/** 
 * 64-bit complex double, vector copy method.
 * \param pSrc \ref source_signal_pointer.
 * \param pDst \ref destination_signal_pointer.
 * \param len \ref length_specification.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsCopy_64fc(const Npp64fc * pSrc, Npp64fc * pDst, int len);

// end of Signal Copy
//@}


/** @name Statistical Functions
 * Functions that provide global signal statistics like: average, standard
 * deviation, minimum, etc.
 */
//@{

/** 
 * Device-buffer size (in bytes) for 8u reductions.
 * This primitive provides the correct buffer size for nppsSum_8u, 
 * nppsMin_8u, nppsMax_8u.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_8u(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 16s reductions.
 * This primitive provides the correct buffer size for nppsSum_16s, 
 * nppsMin_16s, nppsMax_16s.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_16s(int nLength, int * hpBufferSize /* host pointer */);

/** 
 * Device-buffer size (in bytes) for 16u reductions.
 * This primitive provides the correct buffer size for nppsSum_16u, 
 * nppsMin_16u, nppsMax_16u.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
inline
NppStatus 
nppsReductionGetBufferSize_16u(int nLength, int * hpBufferSize /* host pointer */)
{
    return nppsReductionGetBufferSize_16s(nLength, hpBufferSize);
}

/** 
 * Device-buffer size (in bytes) for 16s reductions with integer-results scaling.
 * This primitive provides the correct buffer size for nppsSum_16s_Sfs.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_16s_Sfs(int nLength, int * hpBufferSize /* host pointer */);

/** 
 * Device-buffer size (in bytes) for 16sc reductions.
 * This primitive provides the correct buffer size for nppsSum_16sc, 
 * nppsMin_16sc, nppsMax_16sc.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_16sc(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 16sc reductions with integer-results scaling.
 * This primitive provides the correct buffer size for nppsSum_16sc_Sfs.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_16sc_Sfs(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 32s reductions.
 * This primitive provides the correct buffer size for nppsSum_32sc, 
 * nppsMin_32sc, nppsMax_32sc.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_32s(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 32u reductions.
 * This primitive provides the correct buffer size for nppsSum_32u, 
 * nppsMin_32u, nppsMax_32u.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
inline
NppStatus 
nppsReductionGetBufferSize_32u(int nLength, int * hpBufferSize /* host pointer */)
{
    return nppsReductionGetBufferSize_32s(nLength, hpBufferSize);
}
 
/** 
 * Device-buffer size (in bytes) for 32s reductions with integer-results scaling.
 * This primitive provides the correct buffer size for nppsSum_32s_Sfs.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> 
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_32s_Sfs(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 32sc reductions.
 * This primitive provides the correct buffer size for nppsSum_32sc, 
 * nppsMin_32sc, nppsMax_32sc.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: hpBufferSize is a 
 *        <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_32sc(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 32f reductions.
 * This primitive provides the correct buffer size for nppsSum_32f, 
 * nppsMin_32f, nppsMax_32f.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: hpBufferSize is a 
 *        <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_32f(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 32fc reductions.
 * This primitive provides the correct buffer size for nppsSum_32fc, 
 * nppsMin_32fc, nppsMax_32fc.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: hpBufferSize is a 
 *        <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_32fc(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 64s reductions.
 * This primitive provides the correct buffer size for nppsSum_64s, 
 * nppsMin_64s, nppsMax_64s.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: hpBufferSize is a 
 *        <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_64s(int nLength, int * hpBufferSize /* host pointer */);
 
/** 
 * Device-buffer size (in bytes) for 64f reductions.
 * This primitive provides the correct buffer size for nppsSum_64f, 
 * nppsMin_64f, nppsMax_64f.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: hpBufferSize is a 
 *        <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_64f(int nLength, int * hpBufferSize /* host pointer */);

/** 
 * Device-buffer size (in bytes) for 64fc reductions.
 * This primitive provides the correct buffer size for nppsSum_64fc, 
 * nppsMin_64fc, nppsMax_64fc.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: hpBufferSize is a 
 *        <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus 
nppsReductionGetBufferSize_64fc(int nLength, int * hpBufferSize /* host pointer */);

/** 
 * 32-bit float vector sum method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param eHint Suggests using specific code.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_32f(const Npp32f * pSrc, int nLength, Npp32f * pSum, NppHintAlgorithm eHint, 
            Npp8u * pDeviceBuffer);

/** 
 * 32-bit float complex vector sum method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param eHint Suggests using specific code.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_32fc(const Npp32fc * pSrc, int nLength, Npp32fc * pSum, NppHintAlgorithm eHint, 
             Npp8u * pDeviceBuffer);

/** 
 * 64-bit double vector sum method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_64f(const Npp64f * pSrc, int nLength, Npp64f * pSum, Npp8u * pDeviceBuffer);

/** 
 * 64-bit double complex vector sum method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_64fc(const Npp64fc * pSrc, int nLength, Npp64fc * pSum, Npp8u * pDeviceBuffer);

/** 
 * 16-bit short vector sum with integer scaling method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \param nScaleFactor Integer-result scale factor.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_16s_Sfs(const Npp16s * pSrc, int nLength, Npp16s * pSum, int nScaleFactor, 
                Npp8u * pDeviceBuffer);

/** 
 * 32-bit integer vector sum with integer scaling method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \param nScaleFactor Integer-result scale factor.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_32s_Sfs(const Npp32s * pSrc, int nLength, Npp32s * pSum, int nScaleFactor, 
                Npp8u * pDeviceBuffer);

/** 
 * 16-bit short complex vector sum with integer scaling method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \param nScaleFactor Integer-result scale factor.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_16sc_Sfs(const Npp16sc * pSrc, int nLength, Npp16sc * pSum, int nScaleFactor, 
                 Npp8u * pDeviceBuffer);

/** 
 * 16-bit short complex vector sum (32bit int complex) with integer scaling
 * method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \param nScaleFactor Integer-result scale factor.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_16sc32sc_Sfs(const Npp16sc * pSrc, int nLength, Npp32sc * pSum, int nScaleFactor, 
                     Npp8u * pDeviceBuffer);

/** 
 * 16-bit integer vector sum (32bit) with integer scaling method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pSum Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \param nScaleFactor Integer-result scale factor.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsSum_16s32s_Sfs(const Npp16s * pSrc, int nLength, Npp32s * pSum, int nScaleFactor,
                   Npp8u * pDeviceBuffer);

/** 
 * 16-bit integer vector max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMax Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMax_16s(const Npp16s * pSrc, int nLength, Npp16s * pMax, Npp8u * pDeviceBuffer);

/** 
 * 32-bit integer vector max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMax Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMax_32s(const Npp32s * pSrc, int nLength, Npp32s * pMax, Npp8u * pDeviceBuffer);

/** 
 * 32-bit float vector max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMax Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMax_32f(const Npp32f * pSrc, int nLength, Npp32f * pMax, Npp8u * pDeviceBuffer);

/** 
 * 64-bit float vector max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMax Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMax_64f(const Npp64f * pSrc, int nLength, Npp64f * pMax, Npp8u * pDeviceBuffer);

/** 
 * 16-bit integer vector min method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMin_16s(const Npp16s * pSrc, int nLength, Npp16s * pMin, Npp8u * pDeviceBuffer);

/** 
 * 32-bit integer vector min method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMin_32s(const Npp32s * pSrc, int nLength, Npp32s * pMin, Npp8u * pDeviceBuffer);

/** 
 * 32-bit integer vector min method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMin_32f(const Npp32f * pSrc, int nLength, Npp32f * pMin, Npp8u * pDeviceBuffer);

/** 
 * 64-bit integer vector min method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMin_64f(const Npp64f * pSrc, int nLength, Npp64f * pMin, Npp8u * pDeviceBuffer);

/** 
 * Device-buffer size (in bytes) for nppsMinMax_8u.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: 
 *        hpBufferSize is a <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus
nppsMinMaxGetBufferSize_8u(int nLength,  int * hpBufferSize);

/** 
 * Device-buffer size (in bytes) for nppsMinMax_16s.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: 
 *        hpBufferSize is a <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus
nppsMinMaxGetBufferSize_16s(int nLength,  int * hpBufferSize);

/** 
 * Device-buffer size (in bytes) for nppsMinMax_16u.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: 
 *        hpBufferSize is a <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus
nppsMinMaxGetBufferSize_16u(int nLength,  int * hpBufferSize);

/** 
 * Device-buffer size (in bytes) for nppsMinMax_32s.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: 
 *        hpBufferSize is a <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus
nppsMinMaxGetBufferSize_32s(int nLength,  int * hpBufferSize);

/** 
 * Device-buffer size (in bytes) for nppsMinMax_32u.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: 
 *        hpBufferSize is a <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus
nppsMinMaxGetBufferSize_32u(int nLength,  int * hpBufferSize);

/** 
 * Device-buffer size (in bytes) for nppsMinMax_32f.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: 
 *        hpBufferSize is a <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus
nppsMinMaxGetBufferSize_32f(int nLength,  int * hpBufferSize);

/** 
 * Device-buffer size (in bytes) for nppsMinMax_64f.
 * \param nLength \ref length_specification.
 * \param hpBufferSize Required buffer size.  Important: 
 *        hpBufferSize is a <em>host pointer.</em>
 * \return NPP_SUCCESS
 */
NppStatus
nppsMinMaxGetBufferSize_64f(int nLength,  int * hpBufferSize);

/** 
 * 8-bit char vector min and max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the min output result.
 * \param pMax Pointer to the max output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMinMax_8u(const Npp8u * pSrc, int nLength, Npp8u * pMin, Npp8u * pMax, 
              Npp8u * pDeviceBuffer);

/** 
 * 16-bit signed short vector min and max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the min output result.
 * \param pMax Pointer to the max output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMinMax_16s(const Npp16s * pSrc, int nLength, Npp16s * pMin, Npp16s * pMax, 
               Npp8u * pDeviceBuffer);

/** 
 * 16-bit unsigned short vector min and max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the min output result.
 * \param pMax Pointer to the max output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMinMax_16u(const Npp16u * pSrc, int nLength, Npp16u * pMin, Npp16u * pMax, 
               Npp8u * pDeviceBuffer);

/** 
 * 32-bit unsigned int vector min and max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the min output result.
 * \param pMax Pointer to the max output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMinMax_32u(const Npp32u * pSrc, int nLength, Npp32u * pMin, Npp32u * pMax, 
               Npp8u * pDeviceBuffer);

/** 
 * 32-bit signed int vector min and max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the min output result.
 * \param pMax Pointer to the max output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMinMax_32s(const Npp32s * pSrc, int nLength, Npp32s * pMin, Npp32s * pMax, 
               Npp8u * pDeviceBuffer);

/** 
 * 32-bit float vector min and max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the min output result.
 * \param pMax Pointer to the max output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMinMax_32f(const Npp32f * pSrc, int nLength, Npp32f * pMin, Npp32f * pMax, 
               Npp8u * pDeviceBuffer);

/** 
 * 64-bit double vector min and max method
 * \param pSrc \ref source_signal_pointer.
 * \param nLength \ref length_specification.
 * \param pMin Pointer to the min output result.
 * \param pMax Pointer to the max output result.
 * \param pDeviceBuffer Pointer to the required device memory allocation.
 * \return \ref signal_data_error_codes, \ref length_error_codes.
 */
NppStatus 
nppsMinMax_64f(const Npp64f * pSrc, int nLength, Npp64f * pMin, Npp64f * pMax, 
               Npp8u * pDeviceBuffer);

// end of Statistical Functions
//@}

// end of Signal Processing module
//@}
 
#ifdef __cplusplus
} // extern "C"
#endif

#endif // NV_NPPS_H
