#!/bin/bash
#=================================
# how many times do we want to iterate one benchmark run ?
numiter=3

# use up to 2^(numrun-1) cores
numrun=11

# silent compilation
silent=yes

# how long do you want to sleep between two steps.
sleeplen=2
#=================================


#=================================
# No more changes below this line,
# unless you know what you are doing
#=================================

THORNLIST=BENCHMARKS/einstein-xirel-ml
CONFIG=xirelbench

# parameter file generator for weak scaling tests
WEAKPARDIR=BENCHMARKS/par/weakpar

# parameter file generator for strong scaling tests
STRONGPARDIR=BENCHMARKS/par/strongpar

printlogo()
{
echo "
============================================
o   o   o--o      o o--o                o
 \ /  o |   |     | |   |               |
  O     O-Oo  o-o | O--o  o-o o-o   o-o O--o
 / \  | |  \  |-' | |   | |-' |  | |    |  |
o   o | o   o o-o o o--o  o-o o  o  o-o o  o
============================================

A suite of weak and strong scaling benchmarks using Einstein Toolkit

April, 2010
"
}

printstep()
{
echo "
============================================
$1 
============================================
"
}


#=================================
# We start here with a nice 
# XiRelBench logo
#=================================

printlogo

if [ -e "`pwd`/par/weakpar/KL1_ML_vacuum_1_3.par" ]; then
  cd ..
  CACTUSHOME=`pwd`
  cd BENCHMARK
  echo
else
  echo "[ERROR] Please go to your Cactus BENCHMARK directory and run ./xirelbench.sh"
  echo
  exit 1
fi


#=================================
# Get users' input to select a 
# machine to run benchmarks
#=================================

printstep "step 1) select a machine you want to use:"

n=1
for machine in `${CACTUSHOME}/simfactory/sim list-mach | awk '{if($6) print $1":"$2*$4":"$4":"$6}'`
do
   machinename[$n]=`echo $machine | awk -F ':' '{print $1}'`
# it seems we will have to specify this explicit if the # of processors we use
# is less than #ppn
   totalcores[$n]=`echo $machine | awk -F ':' '{print $2}'`
   machineppn[$n]=`echo $machine | awk -F ':' '{print $3}'`
   hostname[$n]=`echo $machine | awk -F ':' '{print $4}'`
   let "n=${n} + 1"
done

count=${#hostname[@]}

n=1
for host in "${hostname[@]}"
do
    echo "$n) ${hostname[$n]}"
    let "n = $n + 1"
done
    echo "q) Quit"

while [ $n != 'q' ]; do
  echo -n "Select machine: "
  read n
  if [ $n = 'q' ]; then
    echo "Done !"
    exit 0
  elif [ $n -gt 0 -a $n -le $count ]; then
    MACHINE=${machinename[$n]}
    PPN=${machineppn[$n]}
    CORES=${totalcores[$n]}
    echo "You select to run benchmarks on $hostname[$n]"
    break
  else
    echo "Please select a machine by inputting the corresponding index."
  fi
done

#=================================
# We will try to do a rsync first
# Won't stop if it fails
#=================================

sleep $sleeplen
printlogo
pwd
printstep "step 2) rsync to ${MACHINE}"
${CACTUSHOME}/simfactory/sim sync ${MACHINE}||
{ echo "
[ERROR]: Rsync failed! Please check the sourcebasedir of ${MACHINE}
 in ${CACTUSHOME}/simfactory/mdb.pm and make sure it is correctly
 configured.
  ";
  exit 1
}

#=================================
# Begin to compile on a remote
# machine
#=================================

sleep $sleeplen
printlogo
printstep "step 3) begin compilation"

${CACTUSHOME}/simfactory/sim remote-build ${MACHINE} ${CONFIG} -thornlist=${THORNLIST} ||
{ echo "
[ERROR]: Compilation failed! You can set silent=1 at the very beginning of
 xirelbench.sh to check where the problem is. It is likely that you need to
 make proper changes to the option file BENCHMARKS/sample_gnu_options.cfg
  ;"
  exit 1;
}


#=================================
# Begin to create and submit a job
# on a remote  machine
#=================================

sleep $sleeplen
printlogo
printstep "step 4) submitting jobs"

# now start benchmark runs. we repeat it 3 times by default.
for (( it = 1; it <= ${numiter}; it++ ))
do
  numcore=1
  for (( n = 1; n <= ${numrun}; n++ ))
  do
    simexist=`${CACTUSHOME}/simfactory/sim remote ${MACHINE} list-sim | grep ${CONFIG}_${numcore}_${it}`
    if [[ $simexist ]]; then
      echo -n "[WARNING]: ${CONFIG}_${numcore}_${it} exists already, do you want to clean up ? y)es, n)o, or q)uit "
      while true; do
        read reply
        if [ $reply = 'q' ]; then
          echo "Done !"
          exit 0
        elif [ $reply = 'y' ]; then
          echo "Purging your old simulation ..."
	  ${CACTUSHOME}/simfactory/sim remote ${MACHINE} purge ${CONFIG}_${numcore}_${it}
          echo "Creating a new simulation ..."
          ${CACTUSHOME}/simfactory/sim remote ${MACHINE} create-submit ${CONFIG}_${numcore}_${it} --configuration ${CONFIG} --parfile=${WEAKPARDIR}/KL9_ML_vacuum_${numcore}_${it}.par --procs=${numcore} -ppn=${PPN} --walltime=1:0:0 ||
          { echo "
[ERROR]: job submission failed. It is likely that your simfactory is not
 properly configured. Please report the error together with the
 option file you use to cactusmaint@cactuscode.org.
 ";
          exit 1
          }
          break
        elif [ $reply = 'n' ]; then
          echo "[WARNING]: your old data will be used for analyze the performance"
          break
        else
          echo -n "Please select y)es, n)o, or q)uit "
        fi
      done
    else
      echo "Creating a new simulation ..."
      ${CACTUSHOME}/simfactory/sim remote ${MACHINE} create-submit ${CONFIG}_${numcore}_${it} --configuration ${CONFIG} --parfile=${WEAKPARDIR}/KL9_ML_vacuum_${numcore}_${it}.par --procs=${numcore} -ppn=${PPN} --walltime=1:0:0 ||
      { echo "
[ERROR]: job submission failed. It is likely that your simfactory is not
 properly configured. Please report the error together with the 
 option file you use to cactusmaint@cactuscode.org.
 ";
      exit 1;
      }
    fi
  numcore=`expr $numcore "*" 2`
  if [ $numcore -ge $CORES ]; then
    echo "[WARNING]: the required number of cores for run $n is ${numcore}
while the system has only ${CORES} cores. Enough is enough."
    break
  fi
  done
done

#=================================
# Done !
# print a nice logo
#=================================

printlogo

echo "
Thank you for using XiRelBench benchmark !
"
